<?php

namespace Ignite\Activity\Domain\Rules;

use Exception;
use Ignite\Activity\Contracts\ManagesPropagation;
use Ignite\Activity\Contracts\Rule as RuleContract;
use Ignite\Activity\Entities\Rule;
use Ignite\Activity\Traits\ManagesPropagationTrait;

abstract class Base implements RuleContract, ManagesPropagation, CanBeProcessed
{
    use ManagesPropagationTrait;

    /**
     * The Rule entity instance.
     *
     * @see Rule::instance()
     * @var Rule
     */
    protected $rule;

    /**
     * Whether the instance has processed the apply() method logic yet.
     *
     * @var bool
     */
    protected $processed = false;

    /**
     * The messages generated from processing the apply() method logic.
     *
     * @var array
     */
    protected $messages = [];

    /**
     * The Base Rule instance.
     * @param Rule $rule
     */
    public function __construct(Rule $rule)
    {
        $this->rule = $rule;
    }

    /**
     * @inheritdoc
     */
    public function label()
    {
        return $this->rule->meta('label', class_basename($this));
    }

    /**
     * @inheritdoc
     */
    public function description()
    {
        return $this->rule->meta('description', '');
    }

    /**
     * Set the rule as processed/unprocessed.
     *
     * @param bool $processed
     *
     * @return $this
     */
    protected function setProcessed($processed = true)
    {
        $this->processed = $processed;

        return $this;
    }

    /**
     * @inheritdoc
     */
    public function processed()
    {
        return $this->processed;
    }

    /**
     * @inheritdoc
     */
    public function messages()
    {
        return $this->messages;
    }

    /**
     * Add a message to the stack.
     *
     * @param string $level
     * @param string $message
     *
     * @return $this
     */
    protected function addMessage(string $level, string $message)
    {
        if (! isset($this->messages[$level])) {
            $this->messages[$level] = [];
        }

        $this->messages[$level][] = $message;

        return $this;
    }

    /**
     * Add an info message to the stack.
     *
     * @param string $message
     *
     * @return $this
     */
    protected function addInfoMessage(string $message)
    {
        $this->addMessage('info', $message);

        return $this;
    }

    /**
     * Add an error message to the stack.
     *
     * @param string $message
     *
     * @return $this
     */
    protected function addErrorMessage(string $message)
    {
        $this->addMessage('error', $message);

        return $this;
    }

    /**
     * Add an exception message to the stack.
     *
     * @param Exception $exception
     *
     * @return $this
     */
    protected function addExceptionMessage(Exception $exception)
    {
        $message = sprintf('%s - File: %s, Line: %s, Code: %s, Stacktrace: %s',
            $exception->getMessage(),
            $exception->getFile(),
            $exception->getLine(),
            $exception->getCode(),
            $exception->getTraceAsString()
        );

        $this->addMessage('error', $message);

        return $this;
    }

    /**
     * Add an array of key-value pairs.
     *
     * @param array $messages
     *
     * @return $this
     */
    protected function addMessages(array $messages)
    {
        foreach ($messages as $level => $message) {
            $this->addMessage($level, $message);
        }

        return $this;
    }

    /**
     * Process the rule as part of the pipeline.
     *
     * @param RuleManager $manager
     * @param callable $next
     *
     * @return RuleManager
     */
    public function process(RuleManager $manager, callable $next)
    {
        $value = $manager->value() ?? $manager->getSubmission()->value ?? 0;

        try {
            $result = $this->apply($manager);

            $this->setProcessed(true);

            $manager->addLog(
                $this->rule->getKey(),
                $this->manifest($result, $value, $this->getCustomManifestData())
            );

            if ($this->isPropagationStopped()) {
                return $manager;
            }

            return $next($manager);

        } catch (Exception $exception) {
            $this->addExceptionMessage($exception);

            $this->setProcessed(false);

            $manager->addLog(
                $this->rule->getKey(),
                $this->manifest(false, $value)
            );

            return $manager;
        }
    }

    /**
     * Create the log manifest.
     *
     * @param bool $result
     * @param int|float $value
     * @param array $overrides
     *
     * @return array
     */
    protected function manifest(bool $result, $value, array $overrides = [])
    {
        return array_merge([
            'sequence' => $this->rule->pivot->sequence ?? null,
            'rule_id' => $this->rule->getKey(),
            'rule_name' => $this->rule->related_name,
            'rule_label' => $this->label(),
            'rule_description' => $this->description(),
            'rule_class' => get_class($this),
            'applied' => $result,
            'messages' => $this->messages(),
            'processed' => $this->processed(),
            'propagated' => $this->isPropagationStopped() ? false : true,
            'changes_value' => ($this instanceof ChangesValue),
            'value' => ($this instanceof ChangesValue) ? $this->getChangedValue() : $value,
            'processed_at' => now()->format('Y-m-d H:i:s'),
        ], $overrides);
    }

    /**
     * Provide custom manifest data to merge or override the defaults.
     *
     * @return array
     */
    protected function getCustomManifestData()
    {
        return [];
    }
}
