<?php

namespace Ignite\Activity\Tests\Repositories;

use Ignite\Activity\Domain\Schema\Schema;
use Ignite\Activity\Domain\Submissions\ActivitySubmissionException;
use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Resource;
use Ignite\Activity\Entities\Rule;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Entities\SubmissionStatus;
use Ignite\Activity\Entities\Type;
use Ignite\Activity\Contracts\ActivitySubmissionRepository as ActivitySubmissionRepositoryContract;
use Ignite\Activity\Events\ActivitySubmissionCreated;
use Ignite\Activity\Events\ActivitySubmissionFailed;
use Ignite\Activity\Events\ActivitySubmissionCreating;
use Ignite\Activity\Repositories\ActivitySubmissionRepository;
use Ignite\Activity\Tests\TestCase;
use Ignite\Core\Entities\User;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Event;
use Mockery;

class ActivitySubmissionRepositoryTest extends TestCase
{
    use RefreshDatabase;

    /**
     * @test
     */
    public function it_can_implements_the_contract()
    {
        $this->assertInstanceOf(
            ActivitySubmissionRepositoryContract::class,
            $this->app->make(ActivitySubmissionRepository::class)
        );
    }

    /**
     * @test
     */
    public function it_can_save_an_activity_submission()
    {
        Event::fake([
            ActivitySubmissionCreating::class,
            ActivitySubmissionCreated::class,
            ActivitySubmissionFailed::class,
        ]);

        $submitter = factory(User::class)->create();

        $data = [
            'company_name' => 'ACME Inc'
        ];

        $schema = array_merge($this->getSchemaData(), ['model' => MockClaim::class]);
        $schemaInstance = new Schema($schema);
        $this->app->instance('claims', $schemaInstance);

        /** @var Type $type */
        $type = $this->buildActivityType($schema);
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        /** @var Rule $rule */
        $rule = factory(Rule::class)->create(['meta' => ['foo' => 'bar']]);
        $offer->rules()->save($rule);

        $repo = new ActivitySubmissionRepository();

        $activity = $repo->create($offer, $data, $submitter);

        $this->assertTrue($activity->type->is($type));
        $this->assertTrue($activity->offer->is($offer));
        $this->assertInstanceOf(Submission::class, $activity->submission);
        $this->assertInstanceOf(SubmissionStatus::class, $activity->submission->state);
        $this->assertInstanceOf(Resource::class, $activity->resource);

        Event::assertDispatched(ActivitySubmissionCreating::class);
        Event::assertDispatched(ActivitySubmissionCreated::class);
        //Event::assertDispatched(ActivityCreated::class);

        Event::assertNotDispatched(ActivitySubmissionFailed::class);
    }

    /**
     * @test
     */
    public function it_will_not_rollback_the_data_commit_when_an_event_listener_throws_an_exception()
    {
        $message = 'Should not stop the data from being saved to the database.';

        $this->app->events->listen(ActivitySubmissionCreated::class, function () use ($message) {
            throw new \DomainException($message);
        });

        $submitter = factory(User::class)->create();

        $data = [
            'company_name' => 'ACME Inc'
        ];

        $schema = array_merge($this->getSchemaData(), ['model' => MockClaim::class]);
        $schemaInstance = new Schema($schema);
        $this->app->instance('claims', $schemaInstance);

        /** @var Type $type */
        $type = $this->buildActivityType($schema);
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);

        $repo = new ActivitySubmissionRepository();

        try {
            $activity = $repo->create($offer, $data, $submitter);
        } catch (\DomainException $e) {
            $this->assertEquals($message, $e->getMessage());
        }

        $this->assertDatabaseHas('activity_resource', [
            'type_id' => $type->getKey(),
            'offer_id' => $offer->getKey(),
            'user_id' => $submitter->getKey(),
            'data->company_name' => "ACME Inc",
        ]);
    }

    /**
     * @test
     */
    public function it_will_rollback_the_data_and_dispatch_event_listener_throws_an_exception()
    {
        Event::fake([
            ActivitySubmissionFailed::class,
        ]);

        $submitter = factory(User::class)->create();

        $data = [
            'company_name' => 'Brightspot'
        ];

        $schema = array_merge($this->getSchemaData(), ['model' => MockClaim::class]);
        $schemaInstance = new Schema($schema);
        $this->app->instance('claims', $schemaInstance);

        /** @var Type $type */
        $type = $this->buildActivityType($schema);

        $offer = Mockery::mock(Offer::class)->makePartial();
        $offer->shouldReceive('getKey')->andThrow(\Exception::class, 'getKey() throws an exception.');

        $repo = new ActivitySubmissionRepository();

        try {
            $activity = $repo->create($offer, $data, $submitter);
        } catch (ActivitySubmissionException $e) {
            $this->assertInstanceOf(ActivitySubmissionException::class, $e);
        }

        Event::assertDispatched(ActivitySubmissionFailed::class);

        $this->assertDatabaseMissing('activity_resource', [
            'type_id' => $type->getKey(),
            'user_id' => $submitter->getKey(),
            'data->company_name' => "Brightspot",
        ]);
    }
}

class MockClaim extends Model
{
    protected $guarded = [];

    public function save(array $options = [])
    {
        return $this->attributes;
    }
}
