<?php

namespace Ignite\Activity\Tests;

use Ignite\Activity\Contracts\Rule as RuleContract;
use Ignite\Activity\Domain\Rules\RuleFactory;
use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Rule;
use Ignite\Activity\Entities\RuleLog;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Entities\Type;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Carbon;
use Mockery;

class RuleLogTest extends TestCase
{
    use RefreshDatabase;

    /**
     * @test
     */
    public function it_can_get_the_generated_value_from_the_rule_log()
    {
        /** @var Type $type */
        $type = $this->buildActivityType();
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create([
            'offer_id' => $type->getFirstOffer()->getKey()
        ]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create([
            'activity_id' => $activity->getKey(),
        ]);
        /** @var RuleLog $ruleLog */
        $ruleLog = factory(RuleLog::class)->create([
            'submission_id' => $submission->getKey(),
            'value' => 1000,
            'details' => [
                'rule_id' => 1
            ]
        ]);

        $this->assertEquals(1000, $ruleLog->value());
    }

    /**
     * @test
     */
    public function it_casts_details_from_json_to_array()
    {
        /** @var Type $type */
        $type = $this->buildActivityType();
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create([
            'offer_id' => $type->getFirstOffer()->getKey()
        ]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create([
            'activity_id' => $activity->getKey(),
        ]);
        /** @var RuleLog $ruleLog */
        $ruleLog = factory(RuleLog::class)->create([
            'submission_id' => $submission->getKey(),
            'value' => 1000,
            'details' => [
                'rule_id' => 1
            ]
        ]);

        $this->assertEquals(1, $ruleLog->details['rule_id']);
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_type()
    {
        /** @var Type $type */
        $type = $this->buildActivityType();
        $offer = $type->getFirstOffer();
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create([
            'offer_id' => $offer->getKey()
        ]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create([
            'activity_id' => $activity->getKey(),
        ]);
        /** @var RuleLog $ruleLog */
        $ruleLog = factory(RuleLog::class)->create([
            'type_id' => $type->getKey(),
            'offer_id' => $offer->getKey(),
            'submission_id' => $submission->getKey(),
            'activity_id' => $activity->getKey(),
            'details' => [
                'rule_id' => 1
            ]
        ]);

        $this->assertInstanceOf(Type::class, $ruleLog->type);
        $this->assertTrue($ruleLog->type->is($type));
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_offer()
    {
        /** @var Type $type */
        $type = $this->buildActivityType();
        $offer = $type->getFirstOffer();
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create([
            'offer_id' => $offer->getKey()
        ]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create([
            'activity_id' => $activity->getKey(),
        ]);
        /** @var RuleLog $ruleLog */
        $ruleLog = factory(RuleLog::class)->create([
            'type_id' => $type->getKey(),
            'offer_id' => $offer->getKey(),
            'submission_id' => $submission->getKey(),
            'activity_id' => $activity->getKey(),
            'details' => [
                'rule_id' => 1
            ]
        ]);

        $this->assertInstanceOf(Offer::class, $ruleLog->offer);
        $this->assertTrue($ruleLog->offer->is($offer));
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_activity()
    {
        /** @var Type $type */
        $type = $this->buildActivityType();
        $offer = $type->getFirstOffer();
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create([
            'offer_id' => $offer->getKey()
        ]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create([
            'activity_id' => $activity->getKey(),
        ]);
        /** @var RuleLog $ruleLog */
        $ruleLog = factory(RuleLog::class)->create([
            'type_id' => $type->getKey(),
            'offer_id' => $offer->getKey(),
            'submission_id' => $submission->getKey(),
            'activity_id' => $activity->getKey(),
            'details' => [
                'rule_id' => 1
            ]
        ]);

        $this->assertInstanceOf(Activity::class, $ruleLog->activity);
        $this->assertTrue($ruleLog->activity->is($activity));
    }

    /**
     * @test
     */
    public function it_defines_a_belongs_to_relationship_to_the_submission()
    {
        /** @var Type $type */
        $type = $this->buildActivityType();
        $offer = $type->getFirstOffer();
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create([
            'offer_id' => $offer->getKey()
        ]);
        /** @var Submission $submission */
        $submission = factory(Submission::class)->create([
            'activity_id' => $activity->getKey(),
        ]);
        /** @var RuleLog $ruleLog */
        $ruleLog = factory(RuleLog::class)->create([
            'type_id' => $type->getKey(),
            'offer_id' => $offer->getKey(),
            'submission_id' => $submission->getKey(),
            'activity_id' => $activity->getKey(),
            'details' => [
                'rule_id' => 1
            ]
        ]);

        $this->assertInstanceOf(Submission::class, $ruleLog->submission);
        $this->assertTrue($ruleLog->submission->is($submission));
    }
}
