<?php

namespace Ignite\Activity\Tests;

use Ignite\Activity\Contracts\Rule as RuleContract;
use Ignite\Activity\Domain\Rules\Base;
use Ignite\Activity\Domain\Rules\ChangesValue;
use Ignite\Activity\Domain\Rules\RuleChannel;
use Ignite\Activity\Domain\Rules\RuleFactory;
use Ignite\Activity\Domain\Rules\RuleManager;
use Ignite\Activity\Domain\Rules\RuleProcessor;
use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Rule;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Entities\Type;
use Ignite\Activity\Events\ActivitySubmissionRulesProcessed;
use Ignite\Activity\Events\ActivitySubmissionRulesProcessing;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Event;

class RuleProcessorTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Create a Rule entity and instance.
     *
     * @param array $params
     * @param string $name
     * @param Offer|null $offer
     * @param int $sequence
     * @return Rule
     */
    protected function createRule(array $params = [], string $name = 'ExampleRule', Offer $offer = null, $sequence = 1)
    {
        /** @var RuleFactory $ruleFactory */
        $ruleFactory = app(RuleFactory::class);

        $relatedName = class_basename($name);
        $ruleFactory->register($relatedName, $name);

        /** @var Rule $rule */
        $rule = factory(Rule::class)->create(array_merge([
            'related_name' => $relatedName
        ], $params));

        if (is_null($offer)) {
            /** @var Offer $offer */
            $offer = factory(Offer::class)->create();
        }

        $offer->rules()->save($rule, ['sequence' => $sequence]);

        return $rule->refresh();
    }

    /**
     * @test
     */
    public function it_can_process_a_collection_of_rules()
    {
        Event::fake([
            ActivitySubmissionRulesProcessing::class,
            ActivitySubmissionRulesProcessed::class,
        ]);

        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        $rule = $this->createRule([], PropagatingRuleFake::class, $offer, 2);
        $offer->load('rules');
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create([
            'offer_id' => $offer->getKey()
        ]);
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        $activity->submission()->save($submission);
        $activity->setRelation('offer', $offer);

        $manager = new RuleManager();
        $channel = app(RuleChannel::class);
        $processor = new RuleProcessor($manager, $channel);

        $manager = $processor->process($activity, $submission);

        $this->assertNotEmpty($manager->getLog()['details'], 'The log contains at least one item');
        $this->assertFalse($manager->isChangedValue(), 'The value is not changed.');
        $this->assertFalse($manager->hasChangedValue(), 'The log does not contain a rule which caused a change to the value');

        Event::assertDispatched(ActivitySubmissionRulesProcessing::class);
        Event::assertDispatched(ActivitySubmissionRulesProcessed::class);
    }

    /**
     * @test
     */
    public function it_can_stop_propagation_in_a_collection_of_rules()
    {
        Event::fake([
            ActivitySubmissionRulesProcessing::class,
            ActivitySubmissionRulesProcessed::class,
        ]);

        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        $rule1 = $this->createRule([], NonPropagatingRuleFake::class, $offer, 1);
        $rule2 = $this->createRule([], PropagatingRuleFake::class, $offer, 2);
        $offer->load('rules');

        /** @var Activity $activity */
        $activity = factory(Activity::class)->create([
            'offer_id' => $offer->getKey()
        ]);
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        $activity->setRelation('offer', $offer);

        $manager = new RuleManager();
        $channel = app(RuleChannel::class);
        $processor = new RuleProcessor($manager, $channel);

        $manager = $processor->process($activity, $submission);

        $this->assertCount(1, $manager->getLog()['details']);
        $this->assertFalse($manager->getLog()['details'][$rule1->getKey()]['propagated']);

        Event::assertDispatched(ActivitySubmissionRulesProcessing::class);
        Event::assertDispatched(ActivitySubmissionRulesProcessed::class);
    }

    /**
     * @test
     */
    public function it_can_indicate_that_the_rule_has_changed_the_submission_value()
    {
        Event::fake([
            ActivitySubmissionRulesProcessing::class,
            ActivitySubmissionRulesProcessed::class,
        ]);

        $type = $this->buildActivityType();
        /** @var Offer $offer */
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        $rule = $this->createRule([], ValueChangingRuleFake::class, $offer, 2);
        /** @var Activity $activity */
        $activity = factory(Activity::class)->create([
            'offer_id' => $offer->getKey()
        ]);
        $submission = factory(Submission::class)->create([
            'activity_id' => $activity->getKey(),
            'value' => 50
        ]);
        $activity->setRelation('offer', $offer->load('rules'));

        $manager = new RuleManager();
        $channel = app(RuleChannel::class);
        $processor = new RuleProcessor($manager, $channel);

        $manager = $processor->process($activity, $submission);

        $this->assertCount(1, $manager->getLog()['details'], 'The log contains one item');
        $this->assertTrue($manager->isChangedValue(), 'The value is changed.');
        $this->assertTrue($manager->hasChangedValue(), 'The log contains a rule which caused a change to the value');

        Event::assertDispatched(ActivitySubmissionRulesProcessing::class);
        Event::assertDispatched(ActivitySubmissionRulesProcessed::class);
    }
}

class PropagatingRuleFake extends Base implements RuleContract
{
    public function apply(RuleManager $manager)
    {
        $this->addMessage('error', 'Rule was not applied because reasons.');

        return false;
    }
}

class NonPropagatingRuleFake extends Base implements RuleContract
{
    public function apply(RuleManager $manager)
    {
        $this->stopPropagation();

        $this->addMessage('error', 'Rule was not applied because I stopped propagation.');

        return false;
    }
}

class ValueChangingRuleFake extends Base implements ChangesValue
{
    protected $value;

    public function apply(RuleManager $manager)
    {
        $this->value = $manager->getSubmission()->value + 100;

        $this->addMessage('info', "I want to change the value to: {$this->value}");

        return true;
    }

    public function getChangedValue()
    {
        return $this->value;
    }
}
