<?php

namespace Ignite\Activity\Services\Importers;

use Ignite\Activity\Domain\Submissions\ActivitySubmissionException;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Repositories\ActivitySubmissionRepository;
use Ignite\Core\Contracts\Importable;
use Ignite\Core\Models\Import\Importer as BaseImporter;
use Illuminate\Support\Arr;
use Illuminate\Validation\Validator;
use League\Csv\Exception;
use Throwable;

class ActivityStatusImporter extends BaseImporter implements Importable
{
    const SUBMISSION_COLUMN = 'submission_id';

    /**
     * The required data that must be available in the file.
     *
     * @var array
     */
    protected $validation = [
        'submission_id' => 'required',
        'user_id' => 'required',
        'status' => 'required',
        'notes' => 'nullable',
    ];

    /**
     * The list of "known" fields.
     *
     * @var array
     */
    protected $fields = [
        'submission_id' => ['SubmissionID', 'SubmissionId', 'Submission ID', 'ID'],
        'user_id' => ['UserID', 'UserId', 'User ID', 'User'],
        'status' => ['Status', 'Statuses'],
        'notes' => ['Notes', 'Note'],
    ];

    /**
     * @var ActivitySubmissionRepository
     */
    protected $submissionRepository;

    /**
     * Construct an instance of the Activity Submission importer.
     *
     * @param  ActivitySubmissionRepository  $submissionRepository
     */
    public function __construct(ActivitySubmissionRepository $submissionRepository)
    {
        $this->submissionRepository = $submissionRepository;
    }

    /**
     * Return the column header in the csv file to use in order to identify the user.
     *
     * @return string
     */
    protected function getUserIdentifier()
    {
        return 'user_id';
    }

    /**
     * Prepare any dependencies for each iteration of the import process.
     *
     * @return void
     * @throws \League\Csv\Exception
     */
    public function prepare()
    {
        $this->cacheUsers(static::SUBMISSION_COLUMN, 'id');
        $this->record->deleteLog();
    }

    /**
     * Pre process the users lookup before it gets cached.
     *
     * @param  array  $users
     * @return array
     */
    protected function preProcessCacheableUsers($users)
    {
        return array_map('trim', array_map('strtolower', $users));
    }

    /**
     * Fetch the cacheable user submissions from the database.
     *
     * @param  string $key
     * @param  array  $users
     * @return array
     */
    protected function getCacheableUsers($key, $users)
    {
        return Submission::whereIn($key, $users)->get()
            ->mapWithKeys(function ($user) use ($key) {
                return [
                    $user->user_id => $user->{$key}
                ];
            })
            ->toArray();
    }

    /**
     * Validate a single line.
     *
     * @param  array  $data
     * @return Validator
     */
    public function validate(array $data): Validator
    {
        $isValid = $this->createValidatorInstance($data, $this->validation);

        // @todo: other validations
        // $this->checkUniqueness($data);

        return $isValid;
    }

    /**
     * Transform the data for the given line.
     *
     * @param  array  $line
     * @return array
     */
    public function transform(array $line): array
    {
        $line['submission_id'] = trim($line['submission_id']);
        $line['user_id'] = trim($line['user_id']);
        $line['status'] = trim($line['status']);
        $line['notes'] = trim($line['notes']);

        return $line;
    }

    /**
     * Add a success message to the log file.
     *
     * @param  int  $line
     * @param  array  $data
     * @return string
     */
    public function formatImportMessage(int $line, array $data): string
    {
        return sprintf(
            'Imported line `%s` identified by %s `%s`',
            $line,
            $this->getUserIdentifier(),
            Arr::get($data, static::SUBMISSION_COLUMN, 'Missing')
        );
    }

    /**
     * Add an error message to the log file.
     *
     * @param  int  $line
     * @param  array  $data
     * @param  Throwable  $error
     * @return string
     */
    public function formatRejectMessage(int $line, array $data, Throwable $error): string
    {
        return sprintf(
            'Rejected line `%s` identified by %s `%s` => %s',
            $line,
            $this->getUserIdentifier(),
            Arr::get($data, static::SUBMISSION_COLUMN, 'Missing'),
            $this->formatRejectError($error)
        );
    }

    /**
     * Process the data for a single line.
     *
     * @param  array  $data
     * @return bool
     * @throws ActivitySubmissionException
     */
    public function save(array $data): bool
    {
        $submission = Submission::find($data['submission_id']);
        if (!$submission) {
            return false;
        }

        $this->submissionRepository->changeStatus($submission->id, $data['status'], $data['notes']);

        return $submission instanceof Submission;
    }

    /**
     * Process the data for a single line.
     *
     * @param  array  $data
     * @return bool
     */
    public function drySave(array $data): bool
    {
        if (!array_key_exists($data['user_id'], $this->cache['users'])) {
            $this->cache['users'][$data['user_id']][$data['submission_id']] = uniqid();
        }

        return parent::drySave($data);
    }
}
