<?php

namespace Ignite\Activity\Models\Grid;

use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Entities\Type;
use Ignite\Core\Entities\Filters\QueryPermissionFilters;
use Ignite\Core\Models\Grid\EloquentTable;
use Illuminate\Database\DatabaseManager;
use Kris\LaravelFormBuilder\Fields\FormField;
use Yajra\DataTables\DataTables;

/**
 * @property ?Offer $offer
 * @property ?Type $type
 * @property ?string $status
 */
class ActivitySubmissionTable extends EloquentTable
{
    /**
     * The columns that should always be displayed when using the ReportFormConfiguration trait.
     * @var array
     */
    protected $alwaysDisplay = [
        'action', 'checkbox', 'submission_user_id', 'submission_created_at',
        'participant_first', 'participant_last', 'participant_email',
    ];

    /**
     * @var ActivityFormatter
     */
    protected $formatter;

    /**
     * ActivitySubmissionTable constructor.
     *
     * @param DataTables $datatables
     * @param DatabaseManager $databaseManager
     * @param ActivityFormatter $formatter
     * @param array $params
     */
    public function __construct(
        DataTables        $datatables,
        DatabaseManager   $databaseManager,
        ActivityFormatter $formatter,
        array             $params = []
    ) {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;
    }

    /**
     * Set the parameters for this table.
     *
     * This table cannot function without setting type & offer.
     *
     * @param Type $type
     * @param Offer $offer
     * @param string|null $status
     */
    public function setParams(Type $type, Offer $offer, ?string $status = null)
    {
        $this->with('type', $type);
        $this->with('offer', $offer);
        $this->with('status', $status);
    }

    /**
     * The columns to show.
     *
     * @return array
     * @throws \Exception
     */
    public function columns()
    {
        $fields = collect($this->offer->form()->getFields())
            ->filter(function (FormField $field) {
                return ! in_array($field->getType(), ['button', 'submit', 'textarea']);
            })
            ->mapWithKeys(function (FormField $field) {
                $name = trim(ucwords(str_replace(['_', '.', '-'], ' ', $field->getName())));
                $label = trim($field->getOptions()['label'] ?? '');
                $title = trim($field->getOptions()['label_alias'] ?? '');

                if (empty($title)) {
                    $title = $label;
                }

                if (empty($title)) {
                    $title = $name;
                }

                $key = "activity.data.{$field->getOption('name')}";

                return [
                    $key => [
                        'title' => $title,
                        'name' => $key,
                        'data' => $key,
                        'visible' => $field->getOption('visible', true),
                        'sortable' => $field->getOption('sortable', true),
                        'orderable' => $field->getOption('orderable', true),
                        'exportable' => $field->getOption('exportable', true),
                        'searchable' => $field->getOption('searchable', true),
                    ],
                ];
            });

        return array_merge([
            'checkbox' => $this->checkboxColumn(),
            'action' => $this->actionsColumn(),
            'submission_created_at' => [
                'title' => 'Date',
                'name' => 'activity_submission.created_at',
                'orderable' => true,
                'exportable' => true,
            ],
            'submission_id' => [
                'title' => 'ID',
                'name' => 'activity_submission.id',
                'orderable' => true,
                'exportable' => true,
            ],
            'submission_value' => [
                'title' => 'Value',
                'name' => 'activity_submission.value',
                'data' => 'value',
                'orderable' => true,
                'exportable' => true,
            ],
            'submission_status' => [
                'title' => 'Status',
                'name' => 'activity_submission.status',
                'orderable' => true,
                'exportable' => true,
            ],
            'submission_user_id' => [
                'title' => 'User ID',
                'name' => 'activity_submission.user_id',
                'orderable' => true,
                'exportable' => true,
            ],
            'participant_first' => [
                'title' => 'First',
                'name' => 'participant.first',
                'orderable' => true,
                'exportable' => true,
            ],
            'participant_last' => [
                'title' => 'Last',
                'name' => 'participant.last',
                'orderable' => true,
                'exportable' => true,
            ],
            'participant_email' => [
                'title' => 'E-mail',
                'name' => 'participant.email',
                'orderable' => true,
                'exportable' => true,
            ],
        ], $fields->toArray());
    }

    /**
     * The participants query.
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function query()
    {
        return QueryPermissionFilters::for('activity.submission.all.browse')->apply(
            $this->getActivityQuery($this->type, $this->offer)
        );
    }

    /**
     * Activity query.
     *
     * @param Type $type
     * @param Offer $offer
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    protected function getActivityQuery(Type $type, Offer $offer)
    {
        return Submission::query()
            ->with(['activity', 'activity.data', 'participant', 'state'])
            ->whereHas('activity', function ($q) use ($offer) {
                $q->where('offer_id', $offer->getKey());
            })
            ->when($this->status, function ($q) {
                // We need to be explicit about which table the status field is
                // being used here, because datatables uses this query.
                // When the sorting is done on the table, laravel-datatables may
                // join this query with a participant table, and then the status
                // field would become ambiguous unless we specify.
                $q->where('activity_submission.status', $this->status);
            });
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap()
    {
        return array_merge([
            'checkbox' => [$this->formatter, 'checkbox'],
            'action' => [$this->formatter, 'actions'],
            'submission_id' => [$this->formatter, 'submission'],
            'participant_first' => [$this->formatter, 'first'],
            'participant_last' => [$this->formatter, 'last'],
            'participant_email' => [$this->formatter, 'email'],
            'submission_user_id' => [$this->formatter, 'user'],
            'submission_created_at' => [$this->formatter, 'createdAt'],
            'submission_status' => [$this->formatter, 'status'],
        ], $this->getOfferFormatters());
    }

    /**
     * The custom formatters from the form.
     *
     * @return array
     */
    protected function getOfferFormatters()
    {
        $formatters = $this->offer->form()->formatters(collect($this->offer->form()->getFields()));

        if (empty($formatters)) {
            return [];
        }

        return $formatters;
    }

    /**
     * Get default builder parameters.
     *
     * @return array
     */
    protected function getBuilderParameters()
    {
        $params = [
            'order' => [
                [2, 'desc'],
            ],
        ];

        return parent::getBuilderParameters(
            $params,
            $this->getCheckboxColumnParameters(),
            ...func_get_args()
        );
    }
}
