<?php

namespace Ignite\Activity\Http\Views\Composers;

use Exception;
use Ignite\Activity\Contracts\Rule as RuleContract;
use Ignite\Activity\Domain\Rules\ChangesValue;
use Ignite\Activity\Entities\Rule;
use Ignite\Activity\Entities\RuleLog;
use Illuminate\Support\Collection;
use Illuminate\View\View;

class RuleLogComposer
{
    /**
     * @param View $view
     */
    public function compose(View $view)
    {
        if (! $view->ruleLog) {
            $view->ruleLog = $this->defaultRuleLog();
        }

        $view->with('logs', $this->combineLogs($view->ruleLog, $view->offer->rules));
    }

    /**
     * @return RuleLog
     */
    protected function defaultRuleLog(): RuleLog
    {
        $ruleLog = resolve(RuleLog::class);
        $ruleLog->value = 0;

        return $ruleLog;
    }

    /**
     * @param RuleLog    $ruleLog
     * @param Collection $rules
     *
     * @return array
     */
    protected function combineLogs(RuleLog $ruleLog, Collection $rules): array
    {
        $logs = [];

        $details = $this->convertToRuleDetails($rules, $ruleLog->value)->toArray();

        foreach ($details as $sequence => $detail) {
            if (isset($ruleLog->details[$sequence])) {
                $logs[$sequence] = $ruleLog->details[$sequence];
            } else {
                $logs[$sequence] = $detail;
            }
        }

        return $logs;
    }

    /**
     * @param  Collection $rules
     * @param  int        $value
     */
    protected function convertToRuleDetails(Collection $rules, int $value): Collection
    {
        return $rules->map(function ($rule) use ($value) {
            return $this->convertToRuleDetail($rule, compact('value'));
        })->keyBy('sequence');
    }

    /**
     * Convert the rule to a detail array.
     *
     * @param Rule  $rule
     * @param array $overrides
     *
     * @return array
     */
    protected function convertToRuleDetail(Rule $rule, array $overrides = []): array
    {
        $instance = $rule->instance();

        return array_merge([
            'sequence' => $rule->pivot->sequence ?? null,
            'rule_id' => $rule->getKey(),
            'rule_name' => $rule->related_name,
            'rule_label' => $instance->label(),
            'rule_description' => $instance->description(),
            'rule_class' => get_class($instance),
            'applied' => false,
            'messages' => [],
            'processed' => false,
            'propagated' => false,
            'changes_value' => ($instance instanceof ChangesValue),
            'value' => 0,
            'processed_at' => now()->format('Y-m-d H:i:s'),
        ], $overrides);
    }
}
