<?php

namespace Ignite\Activity\Providers;

use Ignite\Activity\Contracts\Schema;
use Ignite\Activity\Domain\Schema\Manager;
use Ignite\Activity\Domain\Offers\OfferFormFactory;
use Ignite\Activity\Domain\Rules\RuleFactory;
use Ignite\Activity\Contracts\ActivitySubmissionRepository as ActivitySubmissionRepositoryContract;
use Ignite\Activity\Http\Forms\OfferForm;
use Ignite\Activity\Repositories\ActivitySubmissionRepository;
use Illuminate\Filesystem\Filesystem;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\View;
use Illuminate\Support\ServiceProvider;
use Illuminate\Database\Eloquent\Factory;
use Kris\LaravelFormBuilder\FormBuilder;

class ActivityServiceProvider extends ServiceProvider
{
    /**
     * @var string
     */
    protected $moduleName = 'Activity';

    /**
     * @var string
     */
    protected $moduleNameLower = 'activity';

    /**
     * Boot the application events.
     *
     * @return void
     */
    public function boot()
    {
        $this->bootTranslations();
        $this->bootConfig();
        $this->bootViews();
        $this->bootFactories();

        $this->loadMigrationsFrom(module_path($this->moduleName, 'database/migrations'));

        View::composer(
            'Activity::admin.submissions.partials.rule-log',
            'Ignite\Activity\Http\Views\Composers\RuleLogComposer'
        );
    }

    /**
     * Register config.
     *
     * @return void
     */
    protected function bootConfig()
    {
        $this->publishes([
            module_path($this->moduleName, 'config/config.php') => config_path($this->moduleNameLower . '.php'),
        ], "ignite-{$this->moduleNameLower}-config");

        $this->mergeConfigFrom(
            module_path($this->moduleName, 'config/config.php'),
            $this->moduleNameLower
        );
    }

    /**
     * Register views.
     *
     * @return void
     */
    public function bootViews()
    {
        $viewPath = resource_path("views/modules/{$this->moduleNameLower}");

        $sourcePath = module_path($this->moduleName, 'resources/views');

        $this->publishes([
            $sourcePath => $viewPath
        ], ['views', "ignite-{$this->moduleNameLower}-views"]);

        $this->loadViewsFrom(array_merge($this->getPublishableViewPaths(), [$sourcePath]), $this->moduleName);
    }

    /**
     * Register translations.
     *
     * @return void
     */
    public function bootTranslations()
    {
        $langPath = resource_path('lang/modules/' . $this->moduleName);

        if (is_dir($langPath)) {
            $this->loadTranslationsFrom($langPath, $this->moduleName);
        } else {
            $this->loadTranslationsFrom(module_path($this->moduleName, 'resources/lang'), $this->moduleName);
        }
    }

    /**
     * Register an additional directory of factories.
     *
     * @return void
     */
    public function bootFactories()
    {
        if (! app()->environment('production') && $this->app->runningInConsole()) {
            app(Factory::class)->load(module_path($this->moduleName, 'database/factories'));
        }
    }

    /**
     * The publishable view paths.
     *
     * @return array
     */
    protected function getPublishableViewPaths(): array
    {
        $paths = [];
        foreach ($this->app->config->get('view.paths') as $path) {
            if (is_dir($path . '/modules/' . $this->moduleNameLower)) {
                $paths[] = $path . '/modules/' . $this->moduleNameLower;
            }
        }
        return $paths;
    }



    /**
     * Register the service provider.
     *
     * @return void
     */
    public function register()
    {
        $this->registerRepositories();
        $this->registerOfferFactory();
        $this->registerRuleFactory();
        $this->registerNotificationFactory();
        $this->registerSchemaManager();
        $this->registerSchema();

        $this->app->register(RouteServiceProvider::class);
        $this->app->register(EventServiceProvider::class);
        $this->app->register(ConsoleServiceProvider::class);
    }

    /**
     * Register concrete implementations of Repository contracts.
     */
    protected function registerRepositories()
    {
        $this->app->singleton(ActivitySubmissionRepositoryContract::class, function ($app) {
            return $app->make(ActivitySubmissionRepository::class);
        });
    }

    /**
     * Register the schema manager.
     */
    protected function registerSchemaManager()
    {
        $this->app->singleton(Manager::class, function ($app) {
            return new Manager($app);
        });

        $this->app->alias(Manager::class, 'schema');
    }

    /**
     * Register the factory for offers.
     */
    protected function registerOfferFactory()
    {
        $this->app->singleton(OfferFormFactory::class, function ($app) {
            $schema = $app->config->get('activity.schema', []);
            $formFactory = new OfferFormFactory($app->make(FormBuilder::class));

            if (! empty($schema)) {
                foreach ($schema as $code => $file) {
                    /** @var Schema $schemaInstance */
                    foreach ($app->get($code)->provides('offers') ?? [] as $name => $class) {
                        $formFactory->register($name, $class);
                    }
                }
            }

            return $formFactory;
        });
    }

    /**
     * Register the factory for rules.
     */
    protected function registerRuleFactory()
    {
        $this->app->singleton(RuleFactory::class, function ($app) {
            $schema = $app->config->get('activity.schema', []);
            $ruleFactory = new RuleFactory();

            if (! empty($schema)) {
                foreach ($schema as $code => $file) {
                    foreach ($app->get($code)->provides('rules') ?? [] as $name => $class) {
                        $ruleFactory->register($name, $class);
                    }
                }
            }

            return $ruleFactory;
        });
    }

    /**
     * Register the factory for notifications.
     */
    protected function registerNotificationFactory()
    {
        // TODO: Implement NotificationFactory
    }

    /**
     * Register the schema instances into the container.
     */
    protected function registerSchema()
    {
        $this->app->booted(function ($app) {
            /** @var Manager $manager */
            $manager = $app->make(Manager::class);
            $schema = $app->config->get('activity.schema', []);

            if (! empty($schema)) {
                foreach ($schema as $code => $file) {
                    $app->singleton($code, function () use ($file, $manager) {
                        return $manager->createFromFile($file);
                    });
                }
            }
        });
    }

    /**
     * Get the services provided by the provider.
     *
     * @return array
     */
    public function provides()
    {
        return [];
    }
}
