<?php

namespace Ignite\Activity\Domain\Rules;

use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Rule;
use Ignite\Activity\Entities\RuleLog;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Events\ActivitySubmissionRulesProcessed;
use Ignite\Activity\Events\ActivitySubmissionRulesProcessing;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class RuleProcessor
{
    /**
     * @var RuleManager
     */
    protected $manager;

    /**
     * @var RuleChannel
     */
    protected $channel;

    /**
     * @var bool
     */
    protected $dryRun = false;

    /**
     * RuleProcessor constructor.
     *
     * @param RuleManager $manager
     * @param RuleChannel $channel
     */
    public function __construct(RuleManager $manager, RuleChannel $channel)
    {
        $this->manager = $manager;
        $this->channel = $channel;
    }

    /**
     * Indicate whether we are doing a dry run or not.
     *
     * @param bool $flag
     *
     * @return $this
     */
    public function setDryRun($flag)
    {
        $this->dryRun = $flag;

        return $this;
    }

    /**
     * Determine whether the processor should store data or not.
     *
     * @return bool
     */
    public function isDryRun()
    {
        return $this->dryRun;
    }

    /**
     * Process the rules.
     *
     * @param Activity $activity
     * @param Submission $submission
     *
     * @return RuleManager
     */
    public function process(Activity $activity, Submission $submission)
    {
        $rules = $this->collect($activity);

        if (! $this->isDryRun()) {
            event(new ActivitySubmissionRulesProcessing($this->manager, $activity, $submission, $rules));
        }

        $manager = $this->processRules($activity, $submission, $rules);

        if (! $this->isDryRun()) {
            event(new ActivitySubmissionRulesProcessed($manager, $activity, $submission));
        }

        return $manager;
    }

    /**
     * Process the rules.
     *
     * @param Activity $activity
     * @param Submission $submission
     * @param Collection $rules
     *
     * @return RuleManager
     */
    protected function processRules(Activity $activity, Submission $submission, Collection $rules)
    {
        $manager = $this->channel->pipe(
            $this->manager->setActivity($activity)->setSubmission($submission),
            $rules->toArray()
        );

        if (! $this->isDryRun()) {
            $submission->ruleLog()->updateOrCreate(
                ['submission_id' => $submission->getKey()],
                $manager->getLog()
            );
        }

        return $manager;
    }

    /**
     * Collect the rules into an array of Rule instances.
     *
     * @param Activity $activity
     *
     * @return Collection
     */
    protected function collect(Activity $activity)
    {
        return $activity->offer->rules
            ->sortBy('pivot.sequence', SORT_NUMERIC)
            ->keyBy('pivot.sequence')
            ->map(function (Rule $rule) {
                return $rule->instance();
            });
    }
}
