<?php

namespace Ignite\Activity\Domain\Rules;

use Ignite\Activity\Contracts\ManagesPropagation;
use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Traits\ManagesPropagationTrait;
use Illuminate\Support\Collection;
use Illuminate\Support\Fluent;

class RuleManager implements ManagesPropagation
{
    use ManagesPropagationTrait;

    /**
     * @var array
     */
    protected $log = [];

    /**
     * @var Activity
     */
    protected $activity;

    /**
     * @var Submission
     */
    protected $submission;

    /**
     * Set the activity model.
     *
     * @param Activity $activity
     *
     * @return self
     */
    public function setActivity(Activity $activity)
    {
        $this->activity = $activity;

        return $this;
    }

    /**
     * The activity model.
     *
     * @return Activity
     */
    public function getActivity()
    {
        return $this->activity;
    }

    /**
     * Set the submission model.
     *
     * @param Submission $submission
     *
     * @return self
     */
    public function setSubmission(Submission $submission)
    {
        $this->submission = $submission;

        return $this;
    }

    /**
     * The submission model.
     *
     * @return Submission
     */
    public function getSubmission()
    {
        return $this->submission;
    }

    /**
     * Add the rule log.
     *
     * @param int $id
     * @param array $manifest
     *
     * @return $this
     */
    public function addLog($id, $manifest)
    {
        $this->log[$id] = $manifest;

        return $this;
    }

    /**
     * Get the complete log.
     *
     * @return array
     */
    public function getLog()
    {
        return [
            'type_id' => $this->activity->type->getKey(),
            'offer_id' => $this->activity->offer->getKey(),
            'activity_id' => $this->activity->getKey(),
            'submission_id' => $this->submission->getKey(),
            'details' => $this->log,
            'value' => (float) $this->value(),
        ];
    }

    /**
     * Is the value actually different to the one on the model.
     *
     * @return bool
     */
    public function isChangedValue()
    {
        return (float) $this->value() !== (float) $this->submission->value;
    }

    /**
     * Determines if the value has changed.
     *
     * @return bool
     */
    public function hasChangedValue()
    {
        return (bool) $this->processedWithChangedValue()->count();
    }

    /**
     * The last value of the log from only Rules that changed the value and were processed.
     *
     * @param int $default
     *
     * @return int|float
     */
    public function value($default = 0)
    {
        return optional($this->processedWithChangedValue()->last())->value ?? $default;
    }

    /**
     * Filtered to include only Rules that changed the value and were processed.
     *
     * @return Collection
     */
    protected function processedWithChangedValue()
    {
        return $this->logCollection()->filter(function ($log) {
            return $log->processed && $log->changes_value;
        });
    }

    /**
     * The log array as a collection.
     *
     * @return Collection
     */
    protected function logCollection()
    {
        $collection = new Collection($this->log);

        return $collection->mapInto(Fluent::class);
    }
}
