<?php

namespace Ignite\Activity\Services\Importers;

use Ignite\Activity\Domain\Submissions\ActivitySubmissionException;
use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Repositories\ActivitySubmissionRepository;
use Ignite\Core\Contracts\Importable;
use Ignite\Core\Models\Import\Importer as BaseImporter;
use Illuminate\Support\Arr;
use Illuminate\Validation\Validator;
use Throwable;

class ActivityStatusImporter extends BaseImporter implements Importable
{
    /**
     * The required data that must be available in the file.
     *
     * @var array
     */
    protected $validation = [
        'activity_id' => 'required',
        'status' => 'required',
    ];

    /**
     * The list of "known" fields.
     *
     * @var array
     */
    protected $fields = [
        'activity_id' => ['ActivityID', 'ActivityId', 'Activity ID', 'ID'],
        'status' => ['Status'],
    ];

    /**
     * @var ActivitySubmissionRepository
     */
    protected $submissionRepository;

    /**
     * Construct an instance of the Activity Submission importer.
     *
     * @param  ActivitySubmissionRepository  $submissionRepository
     */
    public function __construct(ActivitySubmissionRepository $submissionRepository)
    {
        $this->submissionRepository = $submissionRepository;
    }

    /**
     * Return the column header in the csv file to use in order to identify the user.
     *
     * @return string
     */
    protected function getUserIdentifier()
    {
        return 'activity_id';
    }

    /**
     * Display the information text as html while previewing the data.
     *
     * @return string
     */
    public function html(): string
    {
        return '';

        return sprintf(
            '<div class="alert alert-info alert-dismissible">'
            .'<button type="button" class="close" data-dismiss="info" aria-hidden="true">×</button>%s'
            .'</div>',
            'Lorem ipsum dolor sit amet, consectetuer adipiscing elit.'
        );
    }

    /**
     * Prepare any dependencies for each iteration of the import process.
     *
     * @return void
     * @throws \League\Csv\Exception
     */
    public function prepare()
    {
        $this->cacheUsers('activity_id', 'activity_id');
        $this->record->deleteLog();
    }

    /**
     * Pre process the users lookup before it gets cached.
     *
     * @param  array  $users
     * @return array
     */
    protected function preProcessCacheableUsers($users)
    {
        return array_map('trim', array_map('strtolower', $users));
    }

    /**
     * Validate a single line.
     *
     * @param  array  $data
     * @return Validator
     */
    public function validate(array $data): Validator
    {
        $isValid = $this->createValidatorInstance($data, $this->validation);

        // @todo: other validations
        // $this->checkUniqueness($data);

        return $isValid;
    }

    /**
     * Transform the data for the given line.
     *
     * @param  array  $line
     * @return array
     */
    public function transform(array $line): array
    {
        $line['activity_id'] = trim($line['activity_id']);
        $line['status'] = trim($line['status']);

        return $line;
    }

    /**
     * Add a success message to the log file.
     *
     * @param  int  $line
     * @param  array  $data
     * @return string
     */
    public function formatImportMessage(int $line, array $data): string
    {
        return sprintf(
            'Imported line `%s` identified by %s `%s`',
            $line,
            'email',
            Arr::get($data, 'activity_id', 'Missing')
        );
    }

    /**
     * Add an error message to the log file.
     *
     * @param  int  $line
     * @param  array  $data
     * @param  Throwable  $error
     * @return string
     */
    public function formatRejectMessage(int $line, array $data, Throwable $error): string
    {
        return sprintf(
            'Rejected line `%s` identified by %s `%s` => %s',
            $line,
            $this->getUserIdentifier(),
            Arr::get($data, 'activity_id', 'Missing'),
            $this->formatRejectError($error)
        );
    }

    /**
     * Process the data for a single line.
     *
     * @param  array  $data
     * @return bool
     * @throws ActivitySubmissionException
     */
    public function save(array $data): bool
    {
        $activity = Activity::find($this->cache['activities'][$data['activity_id']]);
        if (!$activity) {
            return false;
        }

        $activitySubmission = $activity->submission;
        if (array_key_exists($data['activity_id'], $this->cache['activities'])) {
            $this->submissionRepository->changeStatus($activity->submission->id, $data['status']);
        }

        return $activitySubmission instanceof Submission;
    }

    /**
     * Process the data for a single line.
     *
     * @param  array  $data
     * @return bool
     */
    public function drySave(array $data): bool
    {
        if (!array_key_exists($data['activity_id'], $this->cache['activities'])) {
            $this->cache['activities'][$data['activity_id']] = uniqid();
        }

        return parent::drySave($data);
    }
}
