<?php

namespace Ignite\Activity\Generators;

use Illuminate\Support\Arr;
use Illuminate\Support\Str;
use InvalidArgumentException;

class SchemaGenerator extends Generator
{
    /**
     * @var string
     */
    protected $name;

    /**
     * @var string
     */
    protected $target;

    /**
     * @var string
     */
    protected $label;

    /**
     * @var string
     */
    protected $code;

    /**
     * Generate the file using conventions for the data.
     *
     * @param string $path
     * @param array $vars
     *
     * @return array
     */
    public function generate($path, $vars = [])
    {
        $data = $this->createJson($vars);

        $this->filesystem->makeDirectory(dirname($path), 0644, true, true);

        $this->filesystem->put($path, json_encode($data, JSON_PRETTY_PRINT));
        
        return $data;
    }

    /**
     * Create the json data.
     *
     * @param array $vars
     *
     * @return array
     */
    protected function createJson($vars = [])
    {
        $name = Str::studly(Arr::get($vars, 'name'));

        if (! $name) {
            throw new InvalidArgumentException('You must provide a name e.g. claims.');
        }

        $code = Arr::get($vars, 'code', Str::slug($name, '_'));
        $label = Arr::get($vars, 'label', Str::title($name));

        return [
            "name" => $label,
            "model" => Arr::get($vars, 'model', $this->getDefaultModelClass($name)),
            "provides" => [
                "offers" => Arr::get($vars, 'offer', $this->getDefaultOffers($name, $code)),
                "rules" => Arr::get($vars, 'rules', []),
                "notifications" => Arr::get($vars, 'notifications', [])
            ],
            "statuses" => Arr::get($vars, 'statuses', $this->getDefaultStatuses()),
            "fields" => Arr::get($vars, 'fields', $this->getDefaultFields($code))
        ];
    }

    /**
     * The default model class name.
     *
     * @param string $name
     *
     * @return string
     */
    protected function getDefaultModelClass($name)
    {
        return "\\Ignite\\Program\\Activity\\Entities\\{$name}";
    }

    /**
     * The default offers.
     *
     * @param string $name
     * @param string $code
     *
     * @return array
     */
    protected function getDefaultOffers($name, $code)
    {
        return [
            "{$code}.general" => "Ignite\\Program\\Activity\\Offer\\{$name}\\General"
        ];
    }

    /**
     * The default status states and transitions.
     *
     * @return array
     */
    protected function getDefaultStatuses()
    {
        return [
            "states" => [
                "pending" => [
                    "name" => "pending",
                    "type" => "initial",
                    "color" => "yellow",
                    "icon" => "question"
                ],
                "declined" => [
                    "name" => "declined",
                    "type" => "final",
                    "color" => "red",
                    "icon" => "minus"
                ],
                "approved" => [
                    "name" => "approved",
                    "type" => "normal",
                    "color" => "green",
                    "icon" => "check"
                ],
                "cancelled" => [
                    "name" => "cancelled",
                    "type" => "final",
                    "color" => "gray",
                    "icon" => "ban"
                ],
                "issued" => [
                    "name" => "issued",
                    "type" => "final",
                    "color" => "blue",
                    "icon" => "dollar"
                ]
            ],
            "transitions" => [
                "pending" => [
                    "from" => ["pending"],
                    "to" => "pending"
                ],
                "approve" => [
                    "from" => ["pending"],
                    "to" => "approved"
                ],
                "decline" => [
                    "from" => ["pending"],
                    "to" => "declined"
                ],
                "cancel" => [
                    "from" => ["approved"],
                    "to" => "cancelled"
                ],
                "issue" => [
                    "from" => ["approved"],
                    "to" => "issued"
                ]
            ]
        ];
    }

    /**
     * The default fields.
     *
     * @param string $code
     *
     * @return array
     */
    protected function getDefaultFields($code)
    {
        return [
            [
                "table" => "program_{$code}",
                "frontend_type" => "company",
                "backend_type" => "string",
                "form" => [],
                "name" => "partner_id",
                "label" => "Contact Partner Company",
                "label_show" => true,
                "rules" => [
                    "required"
                ]
            ]
        ];
    }
}
