<?php

namespace Ignite\Activity\Entities;

use Exception;
use Ignite\Activity\Facades\Schema;
use Ignite\Activity\Traits\MetableTrait;
use Ignite\Core\Entities\Base;
use Illuminate\Contracts\Auth\Access\Authorizable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;

/**
 * Activity Type Entity
 *
 * @property int $id
 * @property int $type_id
 * @property string $label
 * @property string $code
 * @property array $meta
 * @property Collection $offers
 * @property Rule[] $rules
 * @property Carbon $created_at
 * @property Carbon $updated_at
 */
class Type extends Base
{
    use MetableTrait;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'activity_type';

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'meta' => 'json',
    ];

    /**
     * The singular type label.
     *
     * @return string
     */
    public function singular()
    {
        return Str::singular($this->label);
    }

    /**
     * The singular type label.
     *
     * @return string
     */
    public function plural()
    {
        return Str::plural($this->label);
    }

    /**
     * The schema instance.
     *
     * @return \Ignite\Activity\Contracts\Schema
     */
    public function getSchema()
    {
        return Schema::type($this->code);
    }

    /**
     * The initial state for this activity type.
     *
     * @return array
     */
    public function getInitialStatus()
    {
        return $this->getSchema()->initialStatus();
    }

    /**
     * The initial state name for this activity type.
     *
     * @return string
     * @throws Exception
     */
    public function getInitialStatusName()
    {
        $status = $this->getInitialStatus();

        if (! $status) {
            throw new Exception("Unable to find the initial status for activity type: {$this->label}");
        }

        return $status['name'];
    }

    /**
     * The statuses available for this activity type.
     *
     * @return array
     */
    public function getStatuses()
    {
        return $this->getSchema()->statuses();
    }

    /**
     * Load the offers() relationship if not already loaded.
     *
     * @return $this
     */
    protected function loadOffers()
    {
        if (! $this->relationLoaded('offers')) {
            $this->load('offers');
        }

        return $this;
    }

    /**
     * Determine whether the current model has the offers relationship already loaded with at least one offer.
     *
     * @return bool
     */
    public function hasOffers()
    {
        return $this->loadOffers()->offers->count() > 0;
    }

    /**
     * Determine whether there is only one related offer.
     *
     * @return bool
     */
    public function hasOneOffer()
    {
        return $this->loadOffers()->offers->count() === 1;
    }

    /**
     * Get the first offer in the sequence.
     *
     * @return Offer
     */
    public function getFirstOffer()
    {
        return $this->loadOffers()->offers->first();
    }

    /**
     * The relationship to the collection of offer records.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function offers()
    {
        return $this->hasMany(Offer::class, 'type_id', 'id')->sequential();
    }

    /**
     * A convenience method for checking whether the given user has the given ability for activities of this type.
     *
     * @param string $action
     * @param Authorizable|null $user
     * @return bool
     */
    public function can(string $action, ?Authorizable $user = null): bool
    {
        // We can use ??= in PHP 7.4+
        $user = $user ?? auth()->user();

        return $user->can("activity.submission.$this->code.$action")
            || $user->can("activity.submission.all.$action");
    }

    // Scopes

    /**
     * Scope a query to sort records by sequence in ascending order with NULLs last.
     *
     * @param Builder $query
     */
    public function scopeSequential(Builder $query)
    {
        $query->orderByRaw('-sequence DESC');
    }
}
