<?php

namespace Ignite\Activity\Repositories;

use Ignite\Activity\Contracts\ActivitySubmissionRepository as ActivitySubmissionRepositoryContract;
use Ignite\Activity\Domain\Submissions\ActivitySubmissionException;
use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Entities\SubmissionStatus;
use Ignite\Activity\Events\ActivitySubmissionCreated;
use Ignite\Activity\Events\ActivitySubmissionCreating;
use Ignite\Activity\Events\ActivitySubmissionFailed;
use Ignite\Activity\Events\ActivitySubmissionStatusChanged;
use Ignite\Activity\Events\ActivitySubmissionUpdated;
use Ignite\Activity\Events\ActivitySubmissionUpdating;
use Ignite\Activity\Events\ActivitySubmissionValueChanged;
use Ignite\Core\Contracts\Entities\Participant;
use Ignite\Core\Entities\User;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Fluent;
use Throwable;

class ActivitySubmissionRepository implements ActivitySubmissionRepositoryContract
{
    /**
     * @inheritdoc
     */
    public function create(Offer $offer, array $data, $submitter)
    {
        $data = new Fluent($data);

        event(new ActivitySubmissionCreating($offer, $data, $submitter));

        DB::beginTransaction();

        try {
            /** @var Activity $activity */
            $activity = Activity::query()->create([
                'submitted_by_user_id' => $submitter->getKey(),
                'offer_id' => $offer->getKey(),
            ]);

            /** @var Submission $submission */
            $submission = $activity->submission()->create([
                'user_id' => $submitter->getKey(),
                'status' => $offer->type->getInitialStatusName(),
            ]);

            $submission->states()->create([
                'label' => $offer->type->getInitialStatusName(),
                //'notes' => $offer->type->getInitialStatusNotes(),
                //'user_id' => $submitter->getKey()
            ]);

            $activity->resource()->create([
                'type_id' => $offer->type->getKey(),
                'offer_id' => $offer->getKey(),
                'user_id' => $submitter->getKey(),
                'activity_id' => $activity->getKey(),
                'submission_id' => $submission->getKey(),
                'data' => $data,
                'offer_data' => $offer->toArray(),
                'rule_data' => $offer->rules->toArray(),
            ]);

            $activity->data()->create($data->toArray());

            DB::commit();
        } catch (Throwable $exception) {
            DB::rollBack();

            event(new ActivitySubmissionFailed($exception, $offer, $data, $submitter, null));

            throw new ActivitySubmissionException($exception->getMessage(), null, $exception);
        }

        event(new ActivitySubmissionCreated($activity, $submission, $submitter));

        return $activity;
    }

    /**
     * @inheritdoc
     */
    public function update($submissionId, array $data, $submitter)
    {
        $data = new Fluent($data);

        /** @var Submission $submission */
        $submission = Submission::query()->findOrFail($submissionId);
        $dataEntity = $submission->activity->data->fill($data->toArray());
        $changes = $dataEntity->getDirty();

        event(new ActivitySubmissionUpdating($submission->activity, $submission, $data, $submitter));

        DB::beginTransaction();

        try {
            $dataEntity->save();

            DB::commit();
        } catch (Throwable $exception) {
            DB::rollBack();

            event(new ActivitySubmissionFailed(
                $exception,
                $submission->activity->offer,
                $data,
                $submitter,
                $submission->activity
            ));

            throw new ActivitySubmissionException($exception->getMessage(), null, $exception);
        }

        event(new ActivitySubmissionUpdated($submission->activity, $submission, $submitter, $changes));

        return $submission;
    }

    /**
     * Change the status of the submission.
     *
     * @param int $submissionId
     * @param string $status
     * @param string|null $notes
     * @param User|Participant|Authenticatable|null $changedByUser
     *
     * @return bool
     * @throws ActivitySubmissionException
     */
    public function changeStatus($submissionId, string $status, string $notes = null, $changedByUser = null)
    {
        DB::beginTransaction();

        $changedByUser = $changedByUser ?? auth()->user();

        try {
            /** @var Submission $submission */
            $submission = Submission::query()->findOrFail($submissionId);

            $old = $submission->status;

            // Allow the state machine to handle state/transition validation.
            $submission->fill([
                'status' => $status,
            ]);

            $submission->save();

            // Record the status change date
            $state = new SubmissionStatus([
                'label' => $status,
                'notes' => $notes,
                'changed_by_user_id' => $changedByUser->getKey(),
            ]);

            $submission->state()->save($state);

            DB::commit();
        } catch (Throwable $exception) {
            DB::rollBack();

            throw new ActivitySubmissionException($exception->getMessage(), null, $exception);
        }

        event(new ActivitySubmissionStatusChanged($submission, $old, $status));

        return true;
    }

    /**
     * Change the value of the submission.
     *
     * @param int $submissionId
     * @param int|float $value
     * @param User|Participant|Authenticatable|null $changedByUser
     * @param bool $isManual
     *
     * @return bool
     * @throws ActivitySubmissionException
     */
    public function changeValue($submissionId, $value, $changedByUser = null, bool $isManual = false)
    {
        DB::beginTransaction();

        $changedByUser = $changedByUser ?? auth()->user();

        try {
            /** @var Submission $submission */
            $submission = Submission::query()->findOrFail($submissionId);

            if ($submission->value != $value) {
                $submission->fill([
                    'value' => $value,
                    'value_is_manual' => $isManual,
                ]);

                $submission->save();
            }

            /* This may be a nested transaction from a calling
             * function so we need to commit even if this method
             * makes no changes.
             * -------------------------------------------------- */
            DB::commit();
        } catch (Throwable $exception) {
            DB::rollBack();

            throw new ActivitySubmissionException($exception->getMessage(), null, $exception);
        }

        event(new ActivitySubmissionValueChanged($submission, $value, $changedByUser));

        return true;
    }
}
