<?php

namespace Ignite\Activity\Models\Grid;

use Ignite\Activity\Entities\Submission;
use Ignite\Core\Models\Grid\Formatter;

class ActivityFormatter extends Formatter
{
    /**
     * @param Submission $submission
     * @return string
     */
    public function actions($submission)
    {
        $actions = [];

        if ($submission->activity->type->can('update')) {
            $actions[] = [
                'click' => '',
                'class' => '',
                'icon' => 'pencil',
                'label' => 'Edit',
                'text' => 'text-default',
                'type' => 'default',
                'link' => route('admin.activity.edit', [
                    'type' => $submission->type()->code,
                    'offer' => $submission->offer->code,
                    'submission' => $submission->getKey(),
                    'activity' => $submission->activity->getKey(),
                ]),
            ];
        }

        if ($submission->activity->type->can('browse')) {
            $actions[] = [
                'click' => '',
                'class' => '',
                'icon' => 'eye',
                'label' => 'View',
                'text' => 'text-default',
                'type' => 'default',
                'link' => route('admin.activity.show', [
                    'type' => $submission->type()->code,
                    'offer' => $submission->offer->code,
                    'submission' => $submission->getKey(),
                    'activity' => $submission->activity->getKey(),
                ]),
            ];
        }

        if ($submission->activity->type->can('change-status')) {
            $actions[] = [
                'click' => '',
                'class' => '',
                'icon' => 'expand',
                'label' => 'Change Status',
                'text' => 'text-default',
                'type' => 'default',
                'link' => route('admin.activity.edit', [
                    'type' => $submission->type()->code,
                    'offer' => $submission->offer->code,
                    'submission' => $submission->getKey(),
                    'activity' => $submission->activity->getKey(),
                    'section' => 'status',
                ]),
            ];
        }

        if ($submission->activity->type->can('delete')) {
            $actions[] = [
                'click' => '',
                'class' => '',
                'icon' => 'trash',
                'label' => 'Delete',
                'text' => 'text-danger',
                'type' => 'danger',
                'link' => route('admin.activity.delete', [
                    'type' => $submission->type()->code,
                    'offer' => $submission->offer->code,
                    'submission' => $submission->getKey(),
                    'activity' => $submission->activity->getKey(),
                ]),
            ];
        }

        return $this->buttons($actions);
    }

    /**
     * Format a checkbox for selecting items in the grid.
     *
     * @param object $model
     *
     * @return \Illuminate\Support\HtmlString
     */
    public function checkbox($model)
    {
        if (! $model) {
            return $this->html('');
        }

        return $this->formatCheckbox($model->getKey());
    }

    /**
     * The submission reference.
     *
     * @param object $model
     *
     * @return \Illuminate\Support\HtmlString
     */
    public function submission($model)
    {
        return $model->getKey();
    }

    /**
     * The participant first name.
     *
     * @param object $model
     *
     * I know in some cases, this will be a native string so adding that here. I feel like it may be in all cases and
     * we have HtmlString here from a copy-paste, but I'm not sure.
     * @return \Illuminate\Support\HtmlString|string
     */
    public function first($model)
    {
        return optional($model->participant)->first ?? '';
    }

    /**
     * The participant last name.
     *
     * @param object $model
     *
     * I know in some cases, this will be a native string so adding that here. I feel like it may be in all cases and
     * we have HtmlString here from a copy-paste, but I'm not sure.
     * @return \Illuminate\Support\HtmlString|string
     */
    public function last($model)
    {
        return optional($model->participant)->last ?? '';
    }

    /**
     * The participant email.
     *
     * @param object $model
     *
     * I know in some cases, this will be a native string so adding that here. I feel like it may be in all cases and
     * we have HtmlString here from a copy-paste, but I'm not sure.
     * @return \Illuminate\Support\HtmlString|string
     */
    public function email($model)
    {
        return optional($model->participant)->email ?? '';
    }

    /**
     * The user.
     *
     * @param object $model
     * @return mixed
     */
    public function user($model)
    {
        return optional($model->participant)->getKey();
    }

    /**
     * @param object|Submission $model
     * @return \Illuminate\Support\HtmlString|mixed
     */
    public function status($model)
    {
        $status = parent::status($model);

        $config = $model->getStateMachineConfig();

        if (array_key_exists($model->status, $config['states'])) {
            return $this->html(sprintf(
                '<span class="label bg-%s">%s</span>',
                $config['states'][$model->status]['color'],
                $status
            ));
        }

        return $status;
    }
}
