<?php

namespace Ignite\Activity\Entities;

use Ignite\Activity\Facades\Schema;
use Ignite\Core\Entities\Base;
use Ignite\Core\Entities\Participant;
use Ignite\Core\Entities\User;
use Ignite\Packages\Presenter\Traits\Presentable;
use Ignite\StateMachine\Contracts\StatefulInterface;
use Ignite\StateMachine\FiniteStateMachineTrait;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Carbon;
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

/**
 * Submissions Entity
 *
 * @property int $id
 * @property int $user_id
 * @property int $activity_id
 * @property string $status
 * @property int $value
 * @property boolean $value_is_manual
 * @property string $internal_notes
 * @property SubmissionStatus $state
 * @property Collection $states
 * @property User $user
 * @property Participant $participant
 * @property Offer $offer
 * @property Activity $activity
 * @property Resource $resource
 * @property RuleLog $ruleLog
 * @property Collection $ruleLogs
 * @property Carbon $created_at
 * @property Carbon $updated_at
 * @method \Ignite\Activity\Presenters\Submission present()
 */
class Submission extends Base implements StatefulInterface, AuditableContract
{
    use FiniteStateMachineTrait, Presentable, Auditable;

    /**
     * The class reference of the presenter.
     *
     * @var string
     */
    protected $presenter = \Ignite\Activity\Presenters\Submission::class;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'activity_submission';

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'value' => 'float',
        'value_is_manual' => 'boolean',
    ];

    /**
     * The relationship to the submission status record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function state()
    {
        return $this->hasOne(SubmissionStatus::class, 'submission_id', 'id')->latest();
    }

    /**
     * The relationship to the submission status record in order of oldest first.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function states()
    {
        return $this->hasMany(SubmissionStatus::class, 'submission_id', 'id')->oldest();
    }

    /**
     * The relationship to the submitter's user record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'user_id', 'user_id');
    }

    /**
     * The relationship to the submitter's user record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function participant()
    {
        return $this->belongsTo(Participant::class, 'user_id', 'user_id');
    }

    /**
     * The relationship to the activity record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function activity()
    {
        return $this->belongsTo(Activity::class, 'activity_id', 'id');
    }

    /**
     * The relationship to the resource record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function resource()
    {
        return $this->hasOne(Resource::class, 'submission_id', 'id');
    }

    /**
     * The relationship to the rule log record in order of newest first.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function ruleLog()
    {
        return $this->hasOne(RuleLog::class, 'submission_id', 'id')->latest();
    }

    /**
     * The relationship to the rule log record in order of newest first.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function ruleLogs()
    {
        return $this->hasMany(RuleLog::class, 'submission_id', 'id');
    }

    /**
     * The relationship to the offer record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOneThrough
     */
    public function offer()
    {
        return $this->hasOneThrough(Offer::class, Activity::class, 'id', 'id', 'activity_id', 'offer_id');
    }

    /**
     * The related activity Type.
     *
     * @return Type
     */
    public function type()
    {
        if (! $this->relationLoaded('offer')) {
            $this->load('offer');
        }

        return $this->offer->type;
    }

    /**
     * @inheritdoc
     */
    public function getStateMachineConfig()
    {
        return Schema::type($this->type()->code)->statuses();
    }

    /**
     * @inheritdoc
     */
    public function getStateMachineAttribute()
    {
        return 'status';
    }

    /**
     * The identifier for the record in the audit log.
     *
     * @return mixed
     */
    public static function getAuditFriendlyField()
    {
        return ['activity_id', 'status'];
    }

    /**
     * Gives you audit entries for this submission, based on the submission & activity type, auditable & context types.
     *
     * @return Collection
     */
    public function directAndRelatedAudits(): Collection
    {
        /** @var Builder $query */
        $query = app(config('audit.implementation'))->newQuery();

        $auditableTypes = [
            Submission::class,
            ltrim($this->activity->type->getSchema()->model(), '\\'),
        ];

        return $query
            ->where(function (Builder $query) use ($auditableTypes) {
                $query->where('auditable_id', $this->getKey())
                    ->whereIn('auditable_type', $auditableTypes);
            })
            ->orWhere(function (Builder $query) use ($auditableTypes) {
                $query->where('context_id', $this->getKey())
                    ->whereIn('context_type', $auditableTypes);
            })
            ->get();
    }
}
