<?php

namespace Ignite\Activity\Tests;

use Ignite\Activity\Contracts\Rule as RuleContract;
use Ignite\Activity\Domain\Rules\Base;
use Ignite\Activity\Domain\Rules\RuleFactory;
use Ignite\Activity\Domain\Rules\RuleManager;
use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Rule;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Entities\Type;
use Illuminate\Foundation\Testing\RefreshDatabase;

class RuleFactoryTest extends TestCase
{
    use RefreshDatabase;

    /**
     * Create a Rule entity and instance.
     *
     * @param array $params
     * @param string $name
     * @param Offer|null $offer
     * @return Rule
     */
    protected function createRule(array $params = [], string $name = 'ExampleRule', Offer $offer = null)
    {
        /** @var RuleFactory $ruleFactory */
        $ruleFactory = app(RuleFactory::class);

        $ruleFactory->register($name, RuleFake::class);

        /** @var Rule $rule */
        $rule = factory(Rule::class)->create(array_merge(['related_name' => $name], $params));

        if (is_null($offer)) {
            /** @var Offer $offer */
            $offer = factory(Offer::class)->create();
        }

        $offer->rules()->save($rule);

        return $rule;
    }

    /**
     * @test
     */
    public function it_can_get_the_label_and_description_of_a_rule_instance()
    {
        $rule = $this->createRule([
            'meta' => [
                'label' => $label = 'Offer within date bounds',
                'description' => $description = 'The offer must be within the date bounds.',
                'available_at' => '2021-01-01',
                'available_until' => '2021-12-31',
            ]
        ]);

        $instance = $rule->instance();

        $this->assertEquals($label, $instance->label());
        $this->assertEquals($description, $instance->description());
    }

    /**
     * @test
     */
    public function it_can_add_messages_during_apply_method()
    {
        $type = $this->buildActivityType();
        $offer = factory(Offer::class)->create(['type_id' => $type->getKey()]);
        $activity = factory(Activity::class)->create(['offer_id' => $offer]);
        $submission = factory(Submission::class)->create(['activity_id' => $activity->getKey()]);
        $rule = $this->createRule([], 'ExampleRule', $offer);

        $manager = new RuleManager();

        $instance = $rule->instance(compact('rule'));

        $result = $instance->apply(
            $manager->setActivity($activity)->setSubmission($submission)
        );

        $this->assertFalse($result);
        $this->assertCount(1, $messages = $instance->messages());
        $this->assertArrayHasKey('error', $messages);
        $this->assertContains('Rule was not applied because reasons.', $messages['error']);
    }
}

class RuleFake extends Base implements RuleContract
{
    public function apply(RuleManager $manager)
    {
        $this->addMessage('error', 'Rule was not applied because reasons.');

        return false;
    }
}
