<?php

namespace Ignite\Activity\Entities;

use Ignite\Activity\Facades\Schema;
use Ignite\Core\Entities\Base;
use Ignite\Core\Entities\Participant;
use Ignite\Core\Entities\User;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Carbon;
use OwenIt\Auditing\Contracts\Auditable;

/**
 * Activity Entity
 *
 * @property int $id
 * @property int $submitter_id
 * @property int $offer_id
 * @property array $meta
 * @property User $user
 * @property Participant $participant
 * @property Offer $offer
 * @property Type $type
 * @property Submission $submission
 * @property Resource $resource
 * @property Model $data
 * @property Carbon $created_at
 * @property Carbon $updated_at
 */
class Activity extends Base
{
    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'activity';

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'meta' => 'json'
    ];

    /**
     * The relationship to the submitter's user record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function user()
    {
        return $this->belongsTo(User::class, 'submitter_id', 'user_id');
    }

    /**
     * The relationship to the submitter's user record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function participant()
    {
        return $this->belongsTo(Participant::class, 'submitter_id', 'user_id');
    }

    /**
     * The relationship to the offer record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function offer()
    {
        return $this->belongsTo(Offer::class, 'offer_id', 'id');
    }

    /**
     * The relationship to the offer record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOneThrough
     */
    public function type()
    {
        return $this->hasOneThrough(Type::class, Offer::class, 'id', 'id', 'offer_id', 'type_id');
    }

    /**
     * The relationship to the submission record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function submission()
    {
        return $this->hasOne(Submission::class, 'activity_id', 'id')->where('user_id', $this->submitter_id);
    }

    /**
     * The relationship to the submission records.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function submissions()
    {
        return $this->hasMany(Submission::class, 'activity_id', 'id');
    }

    /**
     * The relationship to the submission record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function resource()
    {
        return $this->belongsTo(Resource::class, 'id', 'activity_id');
    }

    /**
     * The relationship to the data record.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function data()
    {
        $model = Schema::type($this->type->code)->model();

        return $this->hasOne($model, 'activity_id', 'id');
    }
}
