<?php

namespace Ignite\Activity\Repositories;

use Ignite\Activity\Contracts\ActivitySubmissionRepository as ActivitySubmissionRepositoryContract;
use Ignite\Activity\Domain\Submissions\ActivitySubmissionException;
use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Entities\SubmissionStatus;
use Ignite\Activity\Events\ActivitySubmissionCreated;
use Ignite\Activity\Events\ActivitySubmissionCreating;
use Ignite\Activity\Events\ActivitySubmissionFailed;
use Ignite\Activity\Events\ActivitySubmissionStatusChanged;
use Ignite\Activity\Events\ActivitySubmissionUpdated;
use Ignite\Activity\Events\ActivitySubmissionUpdating;
use Ignite\Activity\Events\ActivitySubmissionValueChanged;
use Ignite\Core\Audit\Auditor;
use Ignite\Core\Contracts\Entities\Participant;
use Ignite\Core\Entities\User;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Fluent;
use Throwable;

class ActivitySubmissionRepository implements ActivitySubmissionRepositoryContract
{
    /**
     * @inheritdoc
     */
    public function create(Offer $offer, array $data, $submitter)
    {
        $data = new Fluent($data);

        event(new ActivitySubmissionCreating($offer, $data, $submitter));

        DB::beginTransaction();

        try {
            /** @var Activity $activity */
            $activity = Activity::query()->create([
                'submitter_id' => $submitter->getKey(),
                'offer_id' => $offer->getKey(),
            ]);

            /** @var Submission $submission */
            $submission = $activity->submission()->create([
                'user_id' => $submitter->getKey(),
                'status' => $offer->type->getInitialStatusName(),
            ]);

            $submission->states()->create([
                'label' => $offer->type->getInitialStatusName(),
                //'notes' => $offer->type->getInitialStatusNotes(),
                //'user_id' => $submitter->getKey()
            ]);

            $activity->resource()->create([
                'type_id' => $offer->type->getKey(),
                'offer_id' => $offer->getKey(),
                'user_id' => $submitter->getKey(),
                'activity_id' => $activity->getKey(),
                'submission_id' => $submission->getKey(),
                'data' => $data,
                'offer_data' => $offer->toArray(),
                'rule_data' => $offer->rules->toArray(),
            ]);

            $activity->data()->create($data->toArray());

            DB::commit();
        } catch (Throwable $exception) {
            DB::rollBack();

            event(new ActivitySubmissionFailed($exception, $offer, $data, $submitter, null));

            throw new ActivitySubmissionException($exception->getMessage(), null, $exception);
        }

        event(new ActivitySubmissionCreated($activity, $submission, $submitter));

        return $activity;
    }

    /**
     * @inheritdoc
     */
    public function update($submissionId, array $data, $user)
    {
        $data = new Fluent($data);

        /** @var Submission $submission */
        $submission = Submission::query()->findOrFail($submissionId);

        event(new ActivitySubmissionUpdating($submission->activity, $submission, $data, $user));

        DB::beginTransaction();

        try {
            $submission->activity->data()->update($data->toArray());

            $submission->activity->touch();

            DB::commit();
        } catch (Throwable $exception) {
            DB::rollBack();

            event(new ActivitySubmissionFailed(
                $exception, $submission->activity->offer, $data, $user, $submission->activity
            ));

            throw new ActivitySubmissionException($exception->getMessage(), null, $exception);
        }

        event(new ActivitySubmissionUpdated($submission->activity, $submission, $user));

        return $submission;
    }

    /**
     * Change the status of the submission.
     *
     * @param  int  $submissionId
     * @param  string  $status
     * @param  string|null  $notes
     * @param  User|Participant|Authenticatable|null  $user
     *
     * @return bool
     * @throws ActivitySubmissionException
     */
    public function changeStatus($submissionId, string $status, string $notes = null, User $user = null)
    {
        DB::beginTransaction();

        $user = $user ?? auth()->user();

        try {
            /** @var Submission $submission */
            $submission = Submission::query()->findOrFail($submissionId);

            $old = $submission->status;

            // Allow the state machine to handle state/transition validation.
            $submission->fill([
                'status' => $status
            ]);

            $submission->save();

            // Record the status change date
            $state = new SubmissionStatus([
                'label' => $status,
                'notes' => $notes,
                'user_id' => $user->getKey()
            ]);

            $submission->state()->save($state);

            DB::commit();
        } catch (Throwable $exception) {
            DB::rollBack();

            throw new ActivitySubmissionException($exception->getMessage(), null, $exception);
        }

        event(new ActivitySubmissionStatusChanged($submission, $old, $status));

        return true;
    }

    /**
     * Change the value of the submission.
     *
     * @param  int  $submissionId
     * @param  int|float  $value
     * @param  User|Participant|Authenticatable|null  $user
     * @param  bool  $isManual
     *
     * @return bool
     * @throws ActivitySubmissionException
     */
    public function changeValue($submissionId, $value, User $user = null, bool $isManual = false)
    {
        DB::beginTransaction();

        $user = $user ?? auth()->user();

        try {
            /** @var Submission $submission */
            $submission = Submission::query()->findOrFail($submissionId);

            if ($submission->value != $value) {
                $submission->fill([
                    'value' => $value,
                    'value_is_manual' => $isManual
                ]);

                $submission->save();
            }

            /* This may be a nested transaction from a calling
             * function so we need to commit even if this method
             * makes no changes.
             * -------------------------------------------------- */
            DB::commit();
        } catch (Throwable $exception) {
            DB::rollBack();

            throw new ActivitySubmissionException($exception->getMessage(), null, $exception);
        }

        event(new ActivitySubmissionValueChanged($submission, $value, $user));

        return true;
    }
}
